/**
 * \file: caam_perm.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Secure Data Container
 * \brief : Implementation of permission mapping functions (imx6<->sdc)
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <sys/types.h>
#include <string.h>

#include <sdc.h>
#include <private/sdc_intern.h>
#include <sdc_perm.h>
#include <linux/caam_ee.h>
#include "caam.h"
#include <sdc/arch/errors.h>
#include <sdc/arch/perm.h>

static const struct sdc_perm_to_caam_perm_map sdc_perm_to_caam_perm_mapping[] = {
    {SDC_PERM_ENCRYPT,      CAAM_PERM_DATA_ENCRYPT_BMSK},
    {SDC_PERM_DECRYPT,      CAAM_PERM_DATA_DECRYPT_BMSK},
    {SDC_PERM_SIGN,         CAAM_PERM_DATA_SIGN_BMSK},
    {SDC_PERM_VERIFY,       CAAM_PERM_DATA_VERIFY_BMSK},
    {SDC_PERM_WRAP,         CAAM_PERM_DATA_WRAP_BMSK},
    {SDC_PERM_UNWRAP,       CAAM_PERM_DATA_UNWRAP_BMSK},
    {SDC_PERM_WRAP_KEY,     CAAM_PERM_KEY_WRAP_BMSK},
    {SDC_PERM_UNWRAP_KEY,   CAAM_PERM_KEY_UNWRAP_BMSK},
    {SDC_PERM_DELETE,       CAAM_PERM_KEY_RELEASE_BMSK}
};

static sdc_error_t caam_key_perms_to_sdc_permissions_one_entry(sdc_error_t error_init,
                                                               uint32_t caam_perm,
                                                               sdc_perm_bmsk_t *sdc_perm)
{
    uint32_t handled_bits = 0;
    size_t idx;

    *sdc_perm = SDC_PERM_NONE;

    for (idx = 0;
         idx < sizeof(sdc_perm_to_caam_perm_mapping)/sizeof(struct sdc_perm_to_caam_perm_map);
         idx++) {
        if (caam_perm & sdc_perm_to_caam_perm_mapping[idx].caam_perm_bit) {
            *sdc_perm |= sdc_perm_to_caam_perm_mapping[idx].sdc_perm_bit;
            handled_bits |= sdc_perm_to_caam_perm_mapping[idx].caam_perm_bit;
        }
    }

    if (handled_bits != caam_perm) {
        error_init = SDC_IMX6_PERMISSION_MISMATCH;
    }

    return error_init;
}

sdc_error_t caam_key_perms_to_sdc_permissions(const struct caam_key_perms *caam_perm,
                                              sdc_permissions_t *sdc_perm)
{
    sdc_error_t ret = SDC_OK;

    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->perms_uid, &sdc_perm->perms_owner);
    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->perms_gid, &sdc_perm->perms_group);
    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->perms_others, &sdc_perm->perms_others);
    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->inheritance_mask_uid, &sdc_perm->perms_inherit_owner);
    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->inheritance_mask_gid, &sdc_perm->perms_inherit_group);
    ret = caam_key_perms_to_sdc_permissions_one_entry(ret, caam_perm->inheritance_mask_others, &sdc_perm->perms_inherit_others);

    return ret;
}

static sdc_error_t sdc_permissions_to_caam_key_perms_one_entry(sdc_error_t error_init,
                                                               sdc_perm_bmsk_t sdc_perm,
                                                               uint32_t *caam_perm)
{
    sdc_perm_bmsk_t handled_bits = 0;
    size_t idx;

    *caam_perm = SDC_PERM_NONE;

    for (idx = 0;
         idx < sizeof(sdc_perm_to_caam_perm_mapping)/sizeof(struct sdc_perm_to_caam_perm_map);
         idx++) {
        if (sdc_perm & sdc_perm_to_caam_perm_mapping[idx].sdc_perm_bit) {
            *caam_perm |= sdc_perm_to_caam_perm_mapping[idx].caam_perm_bit;
            handled_bits |= sdc_perm_to_caam_perm_mapping[idx].sdc_perm_bit;
        }
    }

    if (handled_bits != sdc_perm) {
        error_init = SDC_IMX6_PERMISSION_MISMATCH;
    }

    return error_init;
}

sdc_error_t sdc_permissions_to_caam_key_perms(const sdc_permissions_t *sdc_perm,
                                              struct caam_key_perms *caam_perm)
{
    sdc_error_t ret = SDC_OK;

    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_owner, &caam_perm->perms_uid);
    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_group, &caam_perm->perms_gid);
    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_others, &caam_perm->perms_others);
    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_inherit_owner, &caam_perm->inheritance_mask_uid);
    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_inherit_group, &caam_perm->inheritance_mask_gid);
    ret = sdc_permissions_to_caam_key_perms_one_entry(ret, sdc_perm->perms_inherit_others, &caam_perm->inheritance_mask_others);

    return ret;
}

uint32_t sdc_uid_to_caam_uid(uid_t uid)
{
    if (uid == SDC_FLAG_INVALID_UID)
        return CAAM_KEY_INVALID_UID;

    return (uint32_t)uid;
}

uint32_t sdc_gid_to_caam_gid(gid_t gid)
{
    if (gid == SDC_FLAG_INVALID_GID)
        return CAAM_KEY_INVALID_GID;

    return (uint32_t)gid;
}

uid_t caam_uid_to_sdc_uid(uint32_t uid)
{
    if (uid == CAAM_KEY_INVALID_UID)
        return SDC_FLAG_INVALID_UID;

    return (uid_t)uid;
}

gid_t caam_gid_to_sdc_gid(uint32_t gid)
{
    if (gid == CAAM_KEY_INVALID_GID)
        return SDC_FLAG_INVALID_GID;

    return (gid_t)gid;
}
